<?php
/*
Plugin Name: RDW Auto Info
Description: Zoek op autokenteken via RDW API en krijg autogegevens in het WordPress admin gedeelte.
Version: 1.0
Author: Online? JA! (www.onlineja.nl)
Text Domain: rdw-auto-info
Requires at least: 5.8
Requires PHP: 7.4
Tested up to: 6.8
License: GPLv2 or later
*/

// Version guards (runtime)
if (version_compare(PHP_VERSION, '7.4', '<')) {
    add_action('admin_init', function () {
        deactivate_plugins(plugin_basename(__FILE__));
    });
    add_action('admin_notices', function () {
        echo '<div class="notice notice-error"><p>RDW Auto Info vereist PHP 7.4 of hoger. De plugin is gedeactiveerd.</p></div>';
    });
    return;
}

global $wp_version;
if (version_compare($wp_version, '5.8', '<')) {
    add_action('admin_init', function () {
        deactivate_plugins(plugin_basename(__FILE__));
    });
    add_action('admin_notices', function () {
        echo '<div class="notice notice-error"><p>RDW Auto Info vereist WordPress 5.8 of hoger. De plugin is gedeactiveerd.</p></div>';
    });
    return;
}

if (!defined('RDW_PLUGIN_VERSION')) { define('RDW_PLUGIN_VERSION', '1.0'); }
if (!defined('ABSPATH')) { exit; }

// === RDW telemetry secret — must match the server ===
if (!defined('RDW_TELEMETRY_SECRET')) {
    define('RDW_TELEMETRY_SECRET', 'pR7mXwZQ3jK9v2HfA1eT8bL6sN4yC0uV5qG3rD9pL2mN7xB1tF8aK6zW4vY2hJ0uP9sQ3rL7nD5kA1mT8yC6bF4zR2xW0vN9eJ3gH7qP5lD1cA8sZ6mY4tV2wX0rN9bE3fH7jQ5pL1dS8aZ6xC4vT2yW0nU9mE3kH7rQ5pL1fD8sA6zC4yT2vW0xN9jE3hK7qP5lD1sA8zC6vT4yW2xN0'); // same as server
}

// Helpers
function rdw_fetch_remote_notices() {
    // Verander naar je echte licentiebeheer domein
    $endpoint = 'https://www.onlineja.nl/wp-json/rdw-license/v1/notices';

    // Cache 5 minuten om overbelasting te vermijden
    $cache_key = 'rdw_remote_notices_cache';
    $cached = get_transient($cache_key);
    if (is_array($cached)) {
        return $cached;
    }

    $url = add_query_arg(['secret' => RDW_TELEMETRY_SECRET], $endpoint);
    $response = wp_remote_get($url, ['timeout' => 10]);

    if (is_wp_error($response)) return [];
    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) return [];

    $json = json_decode(wp_remote_retrieve_body($response), true);
    $notices = is_array($json) && !empty($json['notices']) ? $json['notices'] : [];

    set_transient($cache_key, $notices, 5 * MINUTE_IN_SECONDS);
    return $notices;
}

function rdw_show_remote_notices() {
    if (!current_user_can('edit_posts')) return; // toon alleen aan relevante admins/editors

    $notices = rdw_fetch_remote_notices();
    if (!$notices) return;

    foreach ($notices as $n) {
        $level = isset($n['level']) ? $n['level'] : 'info';
        // Map level naar WP notice class
        $class = 'notice ';
        switch ($level) {
            case 'success': $class .= 'notice-success'; break;
            case 'warning': $class .= 'notice-warning'; break;
            case 'error':   $class .= 'notice-error';   break;
            default:        $class .= 'notice-info';    break;
        }
        echo '<div class="' . esc_attr($class) . ' is-dismissible"><p><strong>' . esc_html($n['title'] ?? '') . '</strong><br>' . wp_kses_post($n['message'] ?? '') . '</p></div>';
    }
}
add_action('admin_notices', 'rdw_show_remote_notices');

function rdw_get_site_domain() {
    $home = home_url();
    $host = wp_parse_url($home, PHP_URL_HOST);
    return strtolower($host ?: $home);
}

function rdw_send_log_event($action) {
    // CHANGE to your actual license-manager domain
    $endpoint = 'https://www.onlineja.nl/wp-json/rdw-license/v1/log';

    $body = [
        'action'  => $action,
        'domain'  => rdw_get_site_domain(),
        'version' => RDW_PLUGIN_VERSION,
        'secret'  => RDW_TELEMETRY_SECRET,
    ];

    // Use JSON to avoid form-encoding pitfalls and keep consistent server parsing
    wp_remote_post($endpoint, [
        'timeout' => 10,
        'headers' => ['Content-Type' => 'application/json'],
        'body'    => wp_json_encode($body),
    ]);
}

register_activation_hook(__FILE__, function(){
    rdw_send_log_event('activate');
});

register_deactivation_hook(__FILE__, function(){
    rdw_send_log_event('deactivate');
});

register_uninstall_hook(__FILE__, 'rdw_autoinfo_uninstall');
function rdw_autoinfo_uninstall() {
    rdw_send_log_event('uninstall');
}

require_once plugin_dir_path(__FILE__) . 'includes/helpers.php';
require_once plugin_dir_path(__FILE__) . 'includes/license.php';

function rdw_add_admin_menu() {
    if (current_user_can('edit_posts')) {
		add_menu_page(
			'RDW Auto Info',
			'RDW Auto Info',
			'edit_posts',
			'rdw-auto-info',
			'rdw_admin_page'
		);
    }
}

add_action('admin_menu', 'rdw_add_admin_menu');

function rdw_add_settings_submenu() {
	add_submenu_page(
		'rdw-auto-info',              // parent slug
		'RDW Licentie',               // page title
		'Licentie',                   // menu title
		'manage_options',             // capability
		'rdw-license',                // menu slug
		'rdw_license_settings_page'   // callback
	);
}
add_action('admin_menu', 'rdw_add_settings_submenu');

function rdw_maybe_revalidate_license_daily() {
    // Only in admin, and only once per 24h
    if (!is_admin()) return;

    $last = get_option('rdw_license_last_check', 0);
    if (time() - (int)$last < DAY_IN_SECONDS) return;

    $key = rdw_get_license_key();
    if ($key) {
        rdw_validate_license_remote($key);
        update_option('rdw_license_last_check', time(), false);
    }
}
add_action('admin_init', 'rdw_maybe_revalidate_license_daily');

function rdw_check_for_updates() {
    global $wpdb;
    $table_name = $wpdb->prefix . "rdw_auto_info";

    // Check table existence safely
    $exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
    if ($exists !== $table_name) {
        rdw_install();
    } else {
        // Check if 'created_at' column exists
        $column = $wpdb->get_var( "SHOW COLUMNS FROM " . esc_sql( $table_name ) . " LIKE 'created_at'" );
        if (empty($column)) {
            add_action('admin_notices', 'rdw_show_update_notice');
        }
    }
}
add_action('plugins_loaded', 'rdw_check_for_updates');

function rdw_show_update_notice() {
    echo '<div class="notice notice-warning is-dismissible">';
    echo '<p>' . esc_html__('RDW Auto Info plugin heeft een database-update nodig. Gelieve de plugin te deïnstalleren en opnieuw te installeren.', 'rdw-auto-info') . '</p>';
    echo '</div>';
}


function rdw_install() {
    global $wpdb;
    $table_name = $wpdb->prefix . "rdw_auto_info";
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
    id mediumint(9) NOT NULL AUTO_INCREMENT,
    kenteken varchar(10) NOT NULL,
    merk varchar(100) DEFAULT NULL,
    handelsbenaming varchar(150) DEFAULT NULL,
    bouwjaar int DEFAULT NULL,
    prijs decimal(10,2) DEFAULT NULL,
    data text NOT NULL,
    notes text,
    created_by bigint(20) NOT NULL,
    updated_by bigint(20) NOT NULL,
    created_at datetime DEFAULT CURRENT_TIMESTAMP,
    updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    KEY kenteken (kenteken),
    KEY merk (merk),
    KEY handelsbenaming (handelsbenaming),
    KEY bouwjaar (bouwjaar),
    KEY prijs (prijs)
) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'rdw_install');

function rdw_admin_page() {
    if (current_user_can('edit_posts')) {

        // Forceer realtime validatie van de licentie bij openen van de adminpagina
        if (function_exists('rdw_get_license_key') && function_exists('rdw_validate_license_remote')) {
            $key = rdw_get_license_key();
            if (!empty($key)) {
                rdw_validate_license_remote($key);
            }
        }

        require_once plugin_dir_path(__FILE__) . 'admin-page.php';
    } else {
        echo '<p>' . esc_html__('Je hebt geen toestemming om deze pagina te bekijken.', 'rdw-auto-info') . '</p>';
    }
}

function rdw_enqueue_scripts() {
    wp_enqueue_media();
    wp_enqueue_script(
        'rdw-auto-info-js',
        plugin_dir_url(__FILE__) . 'rdw-auto-info.js',
        array('jquery'),
        RDW_PLUGIN_VERSION,
        true
    );
    wp_enqueue_style(
        'rdw-auto-info-css',
        plugin_dir_url(__FILE__) . 'style.css',
        array(),
        RDW_PLUGIN_VERSION
    );
}
add_action('admin_enqueue_scripts', 'rdw_enqueue_scripts');

function rdw_auto_info_upgrade_schema() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'rdw_auto_info';

    $columns = $wpdb->get_col("DESC $table_name", 0);
    if (!in_array('post_id', $columns, true)) {
        $wpdb->query("ALTER TABLE $table_name ADD COLUMN post_id BIGINT(20) UNSIGNED DEFAULT NULL");
    }
}
add_action('admin_init', 'rdw_auto_info_upgrade_schema');